<?php

namespace App\Modules\Student\Model;

use App\Facades\General;
use App\Models\Account;
use App\Models\Tenant;
use App\Models\User;
use Exception;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Yajra\DataTables\Facades\DataTables;
use DB;

class Student extends Tenant
{
    use SoftDeletes;

    protected $fillable = ['account_id',
        'admission_no',
        'roll_no',
        'class_id',
        'section_id',
        'category_id',
        'house_id',
        'religion',
        'cast',
        'height',
        'weight',
        'measure_date',
        'medical_history',
        'father_name',
        'father_phone',
        'father_occupation',
        'father_photo',
        'mother_name',
        'mother_phone',
        'mother_occupation',
        'mother_photo',
        'guardian_is',
        'guardian_relation',
        'guardian_id',
        'bank_account_no',
        'bank_name',
        'ifsc_code',
        'bank_branch',
        'national_ident_no',
        'local_ident_no',
        'prev_school',
        'note',
        'created_by',
        'updated_by'];

    /**
     * @return BelongsTo
     */
    /*  public function account()
      {
          return $this->belongsTo(Account::class, 'account_id');
      }*/

    /**
     * @return BelongsTo
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id')->withDefault();
    }

    /**
     * @return BelongsTo
     */
    public function studentClass()
    {
        return $this->belongsTo(StudentClass::class, 'class_id')->withDefault();
    }

    /**
     * @return BelongsTo
     */
    public function section()
    {
        return $this->belongsTo(Section::class, 'section_id')->withDefault();
    }

    /**
     * @return BelongsTo
     */
    public function castCategory()
    {
        return $this->belongsTo(CastCategory::class, 'category_id')->withDefault();
    }

    /**
     * @return BelongsTo
     */
    public function studentHouse()
    {
        return $this->belongsTo(StudentHouse::class, 'house_id')->withDefault();
    }

    /**
     * @return BelongsTo
     */
    public function guardian()
    {
        return $this->belongsTo(User::class, 'guardian_id')->withDefault();
    }

    /**
     * return datatable
     *
     * @return JsonResponse
     * @throws Exception
     */
    public static function getStudents()
    {
        $options           = array();
        $options['with']   = ["user", "studentClass", "castCategory"];
        $options['object'] = true;
        $student           = new Student();
        $model             = $student->find($options);

        if (General::isSchoolTeacher()) {
            $staffId = Auth::user()->staff->id;
            $model->whereIn('class_id', function ($query) use ($staffId) {
                $query->select('class_id')->from('class_sections')->where('staff_id', $staffId);
            });
        }

        return DataTables::eloquent($model)
            ->addColumn('name', function (Student $student) {
                return $student->user->name;
            })
            ->addColumn('email', function (Student $student) {
                return $student->user->email;
            })
            ->addColumn('contact_no', function (Student $student) {
                return $student->user->mobile_number;
            })
            ->addColumn('gender', function (Student $student) {
                return $student->user->gender;
            })
            ->addColumn('category', function (Student $student) {
                return $student->castCategory->title;
            })
            ->addColumn('class', function (Student $student) {
                return $student->studentClass->title . ' (' . $student->section->title . ')';
            })
            ->addColumn('action', function (Student $student) {
                return $student->returnAction('students', $student, ['show', 'edit', 'destroy']);
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    /**
     * Return options needed for student
     *
     * @return void
     */
    public static function getAddingStudentOptions()
    {
        $data["studentClasses"] = StudentClass::getStudentClasses();
        $data["sections"]       = Section::getSections();
        $data["gender"]         = General::getGender();
        $data["bloodGroup"]     = General::getBloodGroups();
        $data["categories"]     = CastCategory::getCategories();
        $data["houses"]         = StudentHouse::getHouses();

        return $data;
    }

    /**
     * Save the guardian details
     *
     * @param $request
     * @param bool $update
     * @return User|int|mixed|object|null
     */
    public static function addUpdateGuardianUser($request, $update = false)
    {
        if ($request->guardian_email == '' || $request->guardian_email == null) {
            return null;
        }
        $guardianUser = null;
        if ($update == true) {
            $guardianUser = (new User())->findById($request->guardian_id);
        }
        if ($guardianUser == null) {
            $guardianUser             = new User();
            $guardianUser->created_by = Auth::user()->id;
            $guardianUser->password   = '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi'; // password
        } else {
            $guardianUser->updated_by = Auth::user()->id;
        }

        $guardianUser->account_id = Config::get('account.id');
        $guardianUser->name       = $request->guardian_name;
        $guardianUser->email      = $request->guardian_email;
        $avatar                   = self::uploadStudentAvatars($request, 'guardian_avatar');
        if ($avatar != null) {
            $guardianUser->avatar = $avatar;
        }
        $guardianUser->mobile_number     = $request->guardian_phone;
        $guardianUser->current_address   = $request->guardian_address;
        $guardianUser->permanent_address = $request->guardian_address;
        $guardianUser->save();

        return $guardianUser;
    }

    /**
     * Save student details into user table
     *
     * @param $request
     * @param bool $update
     * @return User|mixed|object|null
     */
    public static function addUpdateStudentUser($request, $update = false)
    {
        $user   = null;
        $insert = false;
        // Request to update data
        if ($update == true) {
            $user = (new User())->findById($request->user_id);
        }
        if ($user == null) {
            $user   = new User();
            $insert = true;
        }
        // match the requested fields with resource
        $user->fill($request->all());
        if ($insert == true) {
            $user->created_by = Auth::user()->id;
            $user->password   = '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi'; // password
        } else {
            $user->updated_by = Auth::user()->id;
        }
        $user->account_id = Config::get('account.id');
        // Upload image
        $avatar = self::uploadStudentAvatars($request, 'avatar');
        if ($avatar != null){
            $user->avatar = $avatar;
        }
        $user->save();

        return $user;
    }

    /**
     * Add student details
     *
     * @param $request
     * @param $user
     * @param $guardianUser
     * @param int $student
     * @return Student|Model|int
     */
    public static function addUpdateStudent($request, $user, $guardianUser, $student = 0)
    {
        $student = (new Student())->findById($student);
        $insert  = false;
        if ($student == null) {
            $student = new Student();
            $insert  = true;

        }
        $student->fill($request->all());
        if ($insert == true) {
            $student->created_by = Auth::user()->id;
        } else {
            $student->updated_by = Auth::user()->id;
        }

        $student->user_id      = $user->id;
        $student->guardian_id  = ($guardianUser != null) ? $guardianUser->id : 0;
        $student->father_photo = self::uploadStudentAvatars($request, 'father_photo');
        $student->mother_photo = self::uploadStudentAvatars($request, 'mother_photo');
        $student->account_id   = Config::get('account.id');
        $student->save();

        return $student;
    }

    /**
     * Upload file on server
     *
     * @param $request
     * @param $fileName
     * @return mixed
     */
    public static function uploadStudentAvatars($request, $fileName)
    {
        if ($request->hasFile($fileName)) {

            return General::uploadAvatar($request, $fileName);
        } else {
            return null;
        }
    }


}
