<?php

namespace App\Modules\Student\Http\Controllers;

use App\Facades\General;
use App\Models\Permission;
use App\Models\Role;
use App\Modules\Student\Model\Section;
use App\Modules\Student\Model\StudentClass;
use App\Modules\Student\Requests\StoreStudent;
use App\Modules\Student\Model\Student;
use Exception;
use Illuminate\Contracts\View\Factory;
use Illuminate\Filesystem\Filesystem;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\User;
use Illuminate\Routing\Redirector;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Storage;
use Illuminate\View\View;
use Throwable;

class StudentController extends Controller
{
    /**
     * Module variable
     *
     * @var array
     */
    protected $moduleName;

    /**
     * Class Constructor.
     *
     * @return string
     */
    public function __construct()
    {
        $this->moduleName = $this->getModuleName();
    }

    /**
     * Display a listing of the resource.
     *
     * @param Request $request
     * @return JsonResponse
     * @throws Exception
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            return Student::getStudents();
        }
        General::userLog('SL000201', ['action_module' => $this->moduleName]);
        return View('student::index', ['moduleName' => $this->moduleName]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Response
     */
    public function create()
    {
        if (!General::isSchoolAdmin() && !General::isStaffStudentACL()) {
            Session::flash('error',
                "You don't have permission to access this module. Kindly contact administration.");
            return redirect('dashboard');
        }
        $data = Student::getAddingStudentOptions();

        return View('student::createOrUpdate', compact('data'))->with('moduleName', $this->moduleName);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreStudent $request
     * @return RedirectResponse
     */
    public function store(StoreStudent $request)
    {
        // Add guardian details
        $guardianUser = Student::addUpdateGuardianUser($request);
        // Add student details into user table
        $user = Student::addUpdateStudentUser($request);
        // Add student details into student table
        $student = Student::addUpdateStudent($request, $user, $guardianUser);

        Session::flash('success',
            trans('locale.success_add_msg', array('module_name' => ucfirst($this->moduleName))));

        General::userLog('SL000302', [
            'action_module' => $this->moduleName,
            'parent_id'     => $student->id,
            'event_data'    => ['name' => $user->name, 'url' => route($this->moduleName . '.edit', $student->id)]
        ]);

        return redirect()->route($this->moduleName . '.index');
    }

    /**
     * Display the specified resource.
     *
     * @param Student $student
     * @return Factory|View
     */
    public function show($student)
    {
        if (!General::isSchoolAdmin() && !General::isStaffStudentACL()) {
            Session::flash('error',
                "You don't have permission to access this module. Kindly contact administration.");
            return redirect('dashboard');
        }

        $options         = array();
        $options['with'] = ["user", "studentClass", "castCategory", "guardian"];
        $student         = (new Student())->findById($student, $options);

        General::userLog('SL000304', [
            'action_module' => $this->moduleName,
            'parent_id'     => $student->id,
            'event_data'    => ['name' => $student->user->name,
                                'url'  => route($this->moduleName . '.show', $student->id)]
        ]);

        return View('student::show', compact('student'))->with('moduleName', $this->moduleName);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Student $student
     * @return Factory|View
     */
    public function edit($student)
    {

        if (!General::isSchoolAdmin() && !General::isStaffStudentACL()) {
            Session::flash('error',
                "You don't have permission to access this module. Kindly contact administration.");
            return redirect('dashboard');
        }

        $student = (new Student())->findById($student);
        $data    = Student::getAddingStudentOptions();

        return View('student::createOrUpdate', compact('student', 'data'))
            ->with('moduleName', $this->moduleName);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Staff $staff
     * @return Factory|View
     */
    public function studentProfile()
    {
        if (!General::isSchoolStudent()) {
            Session::flash('error',
                "You don't have permission to access this module. Kindly contact administration.");
            return redirect('dashboard');
        }
        $student = (new Student())->findById(Auth::user()->student->id);
        $data    = Student::getAddingStudentOptions();

        return View('student::createOrUpdate', compact('student', 'data'))
            ->with('moduleName', $this->moduleName)->with('isStudentProfile', true);
    }
    
	 public function student3DInteractives()
    {
        return View('student::3DInteractives')
            ->with('moduleName', $this->moduleName);
    }
	
	 public function studentBiology()
    {
        return View('student::biology')
            ->with('moduleName', $this->moduleName);
    }
	
	 public function studentChemistry()
    {
        return View('student::chemistry')
            ->with('moduleName', $this->moduleName);
    }
    
	public function studentPhysics()
    {
        return View('student::physics')
            ->with('moduleName', $this->moduleName);
    }

    /**
     * Update student profile
     *
     * @param StoreStudent $request
     * @param $student
     * @return RedirectResponse|Redirector
     */
    public function updateStudentProfile(StoreStudent $request, $student)
    {
        // Add guardian details
        $guardianUser = Student::addUpdateGuardianUser($request, true);
        // Add student details into user table
        $user = Student::addUpdateStudentUser($request, true);
        // Add student details into student table
        $student = Student::addUpdateStudent($request, $user, $guardianUser, $student);

        Session::flash('success',
            trans('locale.success_update_msg', array('module_name' => 'Student Profile')));

        General::userLog('SL000303', [
            'action_module' => 'Student Profile',
            'parent_id'     => $student->id,
            'event_data'    => ['name' => $student->user->name,
                                'url'  => route($this->moduleName . '.edit', $student->id)]
        ]);

        return redirect(route('students.profile'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param StoreStudent $request
     * @param Student $student
     * @return RedirectResponse
     */
    public function update(StoreStudent $request, $student)
    {

        if (!General::isSchoolAdmin() && !General::isStaffStudentACL()) {
            Session::flash('error',
                "You don't have permission to access this module. Kindly contact administration.");
            return redirect('dashboard');
        }

        // Add guardian details
        $guardianUser = Student::addUpdateGuardianUser($request, true);
        // Add student details into user table
        $user = Student::addUpdateStudentUser($request, true);
        // Add student details into student table
        $student = Student::addUpdateStudent($request, $user, $guardianUser, $student);

        Session::flash('success',
            trans('locale.success_update_msg', array('module_name' => ucfirst($this->moduleName))));

        General::userLog('SL000303', [
            'action_module' => $this->moduleName,
            'parent_id'     => $student->id,
            'event_data'    => ['name' => $student->user->name,
                                'url'  => route($this->moduleName . '.edit', $student->id)]
        ]);

        return redirect()->route($this->moduleName . '.index');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param Student $student
     * @return RedirectResponse
     * @throws Exception
     */
    public function destroy($student)
    {
        if (!General::isSchoolAdmin() && !General::isStaffStudentACL()) {
            Session::flash('error',
                "You don't have permission to access this module. Kindly contact administration.");
            return redirect('dashboard');
        }

        $student = (new Student())->findById($student);
        General::userLog('SL000305', [
            'action_module' => $this->moduleName,
            'parent_id'     => $student->id,
            'event_data'    => ['name' => $student->user->name]
        ]);

        Session::flash('success',
            trans('locale.success_delete_msg', array('module_name' => ucfirst($this->moduleName))));
        $student->delete();

        return redirect()->route($this->moduleName . '.index');
    }

    /**
     * Display a listing of the resource.
     *
     * @param Request $request
     * @return false|Factory|View|string
     * @throws Throwable
     */
    public function import(Request $request)
    {
        if ($request->isMethod("post")) {

            $fileResponse = General::uploadImportFile($request, 'file', 'student');
            if ($fileResponse == 'extension_not_match') {
                return json_encode(["success" => false, "message" => "Invalid File Extension."]);
            }

            if ($fileResponse == 'large_file_size') {
                return json_encode(["success" => false,
                                    "message" => "File too large. File must be less than 2MB."]);
            }

            $importDataArr = self::getDataFromImportedFile()['importDataArr'];
            $preview       = view('student::preview', compact('importDataArr'))->render();

            return json_encode(["success" => true, "data" => $preview]);
        } else {
            return View('student::import', ['moduleName' => $this->moduleName]);
        }
    }

    public function importSave(Request $request)
    {
        $importData    = self::getDataFromImportedFile();
        $importDataArr = $importData['importDataArr'];
        $totalData     = $importData['totalData'];
        $totalSaved    = 0;
        foreach ($importDataArr as $key => $data) {

            $studentModel = new Student();
            $userModel    = new \App\Models\User();
            //Check if student exists?
            $userData = $userModel->find([
                "where" => [
                    "name"          => $data['name'],
                    "mobile_number" => $data['mobile'],
                    'role_id'       => Role::ROLE_STUDENT
                ]
            ]);
            if ($userData && count($userData)) {
                $userModel = $userData->first();
            }

            //Save Class if not exists
            $classModel = new StudentClass();
            $class      = $classModel->getStudentClassByTitle($data['grade']);
            if (!$class) {
                $classModel->title      = trim($data['grade']);
                $classModel->account_id = Config::get('account.id');
                $classModel->save();
                $class = $classModel->id;
            }

            //Save Section if not exists
            $sectionModel = new Section();
            $section      = $sectionModel->getSectionByTitle($data['division']);
            if (!$section) {
                $sectionModel->title      = trim($data['division']);
                $sectionModel->account_id = Config::get('account.id');
                $sectionModel->save();
                $section = $sectionModel->id;
            }

            $userModel->name          = trim($data['name']);
            $userModel->mobile_number = trim($data['mobile']);
            $userModel->role_id       = Role::ROLE_STUDENT;
            $userModel->account_id    = Config::get('account.id');
            $userModel->password      = '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi'; // password
            $userModel->save();

            // check the existence of student by user id
            $studentData = $studentModel->find(["where" => ["user_id" => $userModel->id]]);
            if ($studentData && count($studentData)) {
                $studentModel = $studentData->first();
            }
            $studentModel->user_id    = $userModel->id;
            $studentModel->class_id   = $class;
            $studentModel->section_id = $section;
            $studentModel->account_id = Config::get('account.id');
            $studentModel->save();

            $totalSaved++;
        }

        $msg = "File imported successfully. Run imports over $totalData students, out of which $totalSaved are saved.";
        return response()->json(['message' => $msg], 200);
    }

    /**
     * Return data from imported file
     *
     * @return array
     */
    public function getDataFromImportedFile()
    {
        $fileAccess = General::getImportedFileAccess('student');

        // Import CSV to preview
        $filepath = storage_path($fileAccess['path'] . "/" . $fileAccess['filename']);
        // Reading file
        $file = fopen($filepath, "r");

        $importDataArr = [];
        $i             = 0;
        $totalData     = 0;
        while (($fileData = fgetcsv($file, 1000, ",")) !== false) {
            // Skip first row
            if ($i == 0) {
                $i++;
                continue;
            }
            if (!empty($fileData[0]) && $fileData[0] != null && !empty($fileData[1]) && $fileData[1] != null && !empty($fileData[2]) && $fileData[2] != null && !empty($fileData[3]) && $fileData[3] != null) {
                $importDataArr[$i]['name']     = $fileData[0];
                $importDataArr[$i]['grade']    = $fileData[1];
                $importDataArr[$i]['division'] = $fileData[2];
                $importDataArr[$i]['mobile']   = $fileData[3];

                $i++;
            }

            $totalData++;
        }
        fclose($file);

        return [
            'importDataArr' => $importDataArr,
            'totalData'     => $totalData
        ];
    }

    /**
     * Get Module name.
     *
     * @return string
     */
    public function getModuleName()
    {
        return "students";
    }
}
