<?php
require_once __DIR__ . '/../inc/db.php';

class ProductSubcategoryModel {
    private $conn;
    private $uploadDir;

    public function __construct($db_conn) {
        $this->conn = $db_conn;
        $this->uploadDir = __DIR__ . '/../../uploads/subproducts/';
    }

    public function create($data) {
        $slug = $this->generateSlug($data['subcat_name']);

        $stmt = $this->conn->prepare("INSERT INTO product_subcategories (cat_id,subcat_name, description, image, slug) VALUES (?, ?, ?, ?, ?)");
        $stmt->bind_param("issss", $data['cat_id'],$data['subcat_name'], $data['description'], $data['image'], $slug);
        return $stmt->execute();
    }

    public function update($id, $data) {
    // Get existing data
    $existing = $this->getById($id);
    if (!$existing) return false;

    $slug = $this->generateSlug($data['subcat_name']);

    // If new image uploaded
    if (!empty($data['image'])) {
        $this->unlinkOldImage($id);
        $image = $data['image'];
    } else {
        // Retain old image
        $image = $existing['image'];
    }

    $stmt = $this->conn->prepare("UPDATE product_subcategories SET cat_id=?,subcat_name=?, description=?, image=?, slug=? WHERE id=?");
    $stmt->bind_param("issssi", $data['cat_id'],$data['subcat_name'], $data['description'], $image, $slug, $id);
    return $stmt->execute();
}

    public function getById($id) {
        $stmt = $this->conn->prepare("SELECT * FROM product_subcategories WHERE id=?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        return $stmt->get_result()->fetch_assoc();
    }
	
	public function delete($id) {
        $this->unlinkOldImage($id);
        $stmt = $this->conn->prepare("DELETE FROM product_subcategories WHERE id=?");
        $stmt->bind_param("i", $id);
        return $stmt->execute();
    }

    public function toggleStatus($id) {
        $stmt = $this->conn->prepare("UPDATE product_subcategories SET status = NOT status WHERE id=?");
        $stmt->bind_param("i", $id);
        return $stmt->execute();
    }
    
    public function updateSortOrder($id, $sort_order) {
    $stmt = $this->conn->prepare("UPDATE product_subcategories SET sort_order = ? WHERE id = ?");
    $stmt->bind_param("ii", $sort_order, $id);
    return $stmt->execute();
}


    // DELETE OLD IMAGE FILE
    private function unlinkOldImage($id) {
        $category = $this->getById($id);
        if ($category && !empty($category['image'])) {
            $oldImagePath = $this->uploadDir . $category['image'];
            if (file_exists($oldImagePath)) {
                unlink($oldImagePath);
            }
        }
    }

    // Generate slug
    private function generateSlug($subcat_name) {
        return strtolower(preg_replace('/[^A-Za-z0-9-]+/', '-', trim($subcat_name)));
    }
}
