<?php 
ini_set('display_errors', 1);
error_reporting(E_ALL);

include __DIR__ . '/../inc/db.php';
require_once __DIR__ . '/../TCPDF/tcpdf.php';

// =============================
// Helper: Indian Number Format
// =============================
function indian_number_format($num, $decimals = 2) {
    $exploded = explode(".", number_format($num, $decimals, ".", ""));
    $whole = $exploded[0];
    $decimal = isset($exploded[1]) ? $exploded[1] : '';
    $last3 = substr($whole, -3);
    $rest = substr($whole, 0, -3);
    if ($rest != '') {
        $rest = preg_replace("/\B(?=(\d{2})+(?!\d))/", ",", $rest);
        $whole = $rest . "," . $last3;
    }
    return $decimal ? $whole . "." . $decimal : $whole;
}

// =============================
// PDF CLASS
// =============================
class InsuranceInvoicePDF extends TCPDF {
    public function Header() {
        $this->Image(__DIR__ . '/../assets/img/logo.png', 12, 10, 40);
        $this->SetFont('dejavusans', '', 8);
        $this->SetXY(140, 10);
        $this->MultiCell(60, 5,
            "GST No: 27ABCDE1234F1Z5\nMobile: +91 98255 59080\nEmail: mytreasuretrip@gmail.com",
            0, 'R', false
        );
        $this->Line(10, 30, 200, 30);
        $this->Ln(5);
        $this->SetFont('dejavusans', 'B', 13);
        $this->SetTextColor(10, 69, 110);
        $this->Cell(0, 23, 'TRAVEL INSURANCE INVOICE', 0, 1, 'C');
        $this->SetTextColor(0, 0, 0);
        $this->Line(10, 43, 200, 43);
        $this->Ln(4);
    }

    public function Footer() {
        $this->SetY(-25);
        $this->SetFont('dejavusans', '', 8);
        $this->SetTextColor(80, 80, 80);
        $this->Line(10, $this->GetY(), 200, $this->GetY());
        $this->Ln(3);
        $this->MultiCell(0, 5,
            "My Treasure Trip | 17, Suman Sajni Society, Near Jayant Park, Ghodasar, Ahmedabad\n" .
            "Mobile: +91 98255 59080 | Email: mytreasuretrip@gmail.com | GST: 27ABCDE1234F1Z5",
            0, 'C', false
        );
    }

    public function sectionTitle($title) {
        $this->Ln(1);
        $this->SetFont('dejavusans', 'B', 10);
        $this->SetFillColor(12, 69, 110);
        $this->SetTextColor(255, 255, 255);
        $this->Cell(0, 7, "  " . strtoupper($title), 0, 1, 'L', true);
        $this->SetTextColor(0, 0, 0);
        $this->Ln(1);
    }
}

// =============================
// FETCH BOOKING DATA
// =============================
$booking_id = intval($_GET['id'] ?? 0);

$query = "
SELECT b.*, 
       c.name AS customer_name, 
       c.email AS customer_email, 
       c.phone AS customer_phone, 
       c.address AS customer_address
FROM bookings b
LEFT JOIN customers c ON c.id = b.customer_id
WHERE b.id = $booking_id AND b.booking_type = 'insurance'
";
$result = $conn->query($query);
$booking = $result->fetch_assoc();

if (!$booking) {
    die("❌ Insurance booking not found.");
}

// =============================
// CALCULATIONS (Corrected)
// =============================
$rupee = '₹';
$sum_insured   = floatval($booking['insurance_sum_insured']);
$premium_rate  = floatval($booking['premium_rate']);
$gst_rate      = floatval($booking['gst_rate'] ?? 18);
$gst_included  = $booking['gst_included'] ?? 'no';
$paid          = floatval($booking['paid_amount'] ?? 0);
$policy_no     = $booking['policy_no'] ?? '-';
$travelers     = intval($booking['no_of_travelers'] ?? 1);

// --- Duration ---
$start = new DateTime($booking['insurance_start_date']);
$end   = new DateTime($booking['insurance_end_date']);
$duration_days = $end->diff($start)->days + 1;

// --- Base Premium (Depends on Sum Insured, Rate, Duration, and Travelers) ---
$premium_amount = round(($sum_insured * $premium_rate / 100) * $duration_days * $travelers, 2);

// --- GST Logic ---
if ($gst_included === 'yes') {
    // ✅ Add GST on top
    $gst_amount = round(($premium_amount * $gst_rate) / 100, 2);
    $total = $premium_amount + $gst_amount;
} else {
    // ❌ GST not included
    $gst_amount = 0;
    $total = $premium_amount;
}

// --- Balance Calculation ---
$balance = round($total - $paid, 2);

// =============================
// PDF SETUP
// =============================
$pdf = new InsuranceInvoicePDF();
$pdf->SetMargins(10, 50, 10);
$pdf->AddPage();
$pdf->SetFont('dejavusans', '', 9);

// =============================
// CUSTOMER DETAILS
// =============================
$pdf->sectionTitle("Customer & Booking Details");
$html = '
<table cellpadding="5" border="0.5" width="100%">
<tr>
<td width="50%">
<strong>Name:</strong> ' . ucwords($booking['customer_name']) . '<br/>
<strong>Phone:</strong> ' . $booking['customer_phone'] . '<br/>
<strong>Email:</strong> ' . $booking['customer_email'] . '<br/>
<strong>Address:</strong> ' . ucwords($booking['customer_address']) . '
</td>
<td width="50%">
<strong>Booking Code:</strong> ' . $booking['booking_code'] . '<br/>
<strong>Booking Date:</strong> ' . date('d-m-Y', strtotime($booking['booking_date'])) . '<br/>
<strong>Policy No:</strong> ' . htmlspecialchars($policy_no) . '<br/>
<strong>Payment Status:</strong> ' . ucfirst($booking['payment_status']) . '
</td>
</tr>
</table>';
$pdf->writeHTML($html, true, false, false, false, '');

// =============================
// SECTION: INSURANCE DETAILS
// =============================
$pdf->sectionTitle("Insurance Details");
$insurance_html = '
<style>
th { background-color:#f0f8ff; color:#0a456e; font-weight:bold; text-align:center; }
</style>
<table cellpadding="5" border="0.5" width="100%">
<tr>
<th>Insurance Type</th>
<th>Company</th>
<th>Destination</th>
<th>Travel Type</th>
<th>No. of Travellers</th>
</tr>
<tr>
<td align="center">' . strtoupper($booking['insurance_type'] ?? '-') . '</td>
<td align="center">' . strtoupper($booking['insurance_company'] ?? '-') . '</td>
<td align="center">' . strtoupper($booking['insurance_destination'] ?? '-') . '</td>
<td align="center">' . strtoupper($booking['insurance_travel_type'] ?? '-') . '</td>
<td align="center">' . ($booking['no_of_travelers'] ?? '-') . '</td>
</tr>
<tr>
<td colspan="5">
<strong>Coverage:</strong> ' . date('d-m-Y', strtotime($booking['insurance_start_date'])) . 
' to ' . date('d-m-Y', strtotime($booking['insurance_end_date'])) . 
'
</td>
</tr>
</table>';
$pdf->writeHTML($insurance_html, true, false, false, false, '');
// =============================
// INSURANCE SUMMARY
// =============================
$pdf->sectionTitle("Insurance Summary");

$summary_html = '
<table cellpadding="6" border="0.5" width="100%">
<tr><td><strong>Sum Insured</strong></td><td align="right">'.$rupee.' '.indian_number_format($sum_insured).'</td></tr>
<tr><td><strong>Duration</strong></td><td align="right">'.$duration_days.' Days</td></tr>
<tr><td><strong>Premium Rate</strong></td><td align="right">'.$premium_rate.'%</td></tr>
<tr><td><strong>Base Premium</strong></td><td align="right">'.$rupee.' '.indian_number_format($premium_amount).'</td></tr>
<tr><td><strong>GST (18%)</strong></td><td align="right">'.$rupee.' '.indian_number_format($gst_amount).'</td></tr>
<tr style="background-color:#f9f9f9; font-weight:bold; color:#0a456e;">
    <td>Total Premium</td><td align="right">'.$rupee.' '.indian_number_format($total).'</td>
</tr>
<tr><td><strong>Paid Amount</strong></td><td align="right">'.$rupee.' '.indian_number_format($paid).'</td></tr>
<tr style="font-weight:bold; color:#0a456e;">
    <td>Balance Amount</td><td align="right">'.$rupee.' '.indian_number_format($balance).'</td>
</tr>
</table>';
$pdf->writeHTML($summary_html, true, false, false, false, '');

// =============================
// TERMS & CONDITIONS SECTION
// =============================
$pdf->sectionTitle("Terms & Conditions");

$terms = '
<ol style="font-size:8.5pt; line-height:14px;">
<li>This travel insurance policy covers the duration mentioned above only.</li>
<li>Claims are subject to terms and exclusions of the insurance company.</li>
<li>Premium once paid is non-refundable after policy issuance.</li>
<li>Ensure all traveler details are correct before payment confirmation.</li>
<li>My Treasure Trip acts as a facilitating agent and not as the insurer.</li>
<li>All disputes subject to Ahmedabad jurisdiction only.</li>
</ol>';
$pdf->writeHTML($terms, true, false, false, false, '');

// =============================
// OUTPUT PDF
// =============================
$pdf->Output('insurance_invoice_' . $booking['booking_code'] . '.pdf', 'I');
?>
