<?php
ini_set('display_errors', 1);
error_reporting(E_ALL);

include __DIR__ . '/../inc/db.php';
require_once __DIR__ . '/../TCPDF/tcpdf.php';

// ----------------------------
// Format Indian currency numbers
// ----------------------------
function indian_number_format($num, $decimals = 2) {
    $exploded = explode(".", number_format($num, $decimals, ".", ""));
    $whole = $exploded[0];
    $decimal = isset($exploded[1]) ? $exploded[1] : '';

    $last3 = substr($whole, -3);
    $rest = substr($whole, 0, -3);
    if ($rest != '') {
        $rest = preg_replace("/\B(?=(\d{2})+(?!\d))/", ",", $rest);
        $whole = $rest . "," . $last3;
    }

    return $decimal ? $whole . "." . $decimal : $whole;
}
define('INR', '₹');

// ----------------------------
// Custom PDF Class
// ----------------------------
class BookingOrderPDF extends TCPDF {
    public $booking_no;
    public $booking_date;

    public function Header() {
        $this->Image(__DIR__.'/../assets/img/logo.png', 10, 8, 45);
        $this->SetFont('helvetica', '', 8);
        $this->SetXY(140, 8);
        $this->MultiCell(60, 5, 
            "GST No: 27ABCDE1234F1Z5\nMobile: +91 98255 59080\nEmail: mytreasuretrip@gmail.com", 
            0, 'R', false);
        $this->SetDrawColor(12, 69, 110);
        $this->Line(10, 30, 200, 30);
        $this->SetY(32);
        $this->SetFont('helvetica', 'B', 14);
        $this->SetTextColor(10, 69, 110);
        $this->Cell(0, 8, 'BOOKING ORDER', 0, 1, 'C');
        $this->SetTextColor(0, 0, 0);
        $this->Line(10, $this->GetY() + 2, 200, $this->GetY() + 2);
    }

    public function Footer() {
        $this->SetY(-20);
        $this->SetDrawColor(12, 69, 110);
        $this->Line(10, $this->GetY(), 200, $this->GetY());
        $this->Ln(2);
        $this->SetFont('helvetica', '', 8);
        $this->MultiCell(0, 5, 
            "17, Suman Sajni Society, Near Jayant Park, Ghodasar, Ahmedabad, Gujarat, India.\nMobile: +91 98255 59080 | Email: mytreasuretrip@gmail.com | GST No: 27ABCDE1234F1Z5", 
            0, 'C', false);
    }

    public function addSectionTitle($title) {
        if ($this->GetY() > 250) $this->AddPage();
        $this->Ln(2);
        $this->SetFont('helvetica', 'B', 12);
        $this->SetFillColor(12, 69, 110);
        $this->SetTextColor(255, 255, 255);
        $this->Cell(0, 8, "  " . strtoupper($title), 0, 1, 'L', true);
        $this->Ln(2);
        $this->SetTextColor(0, 0, 0);
    }

    // ----------------------------
    // CUSTOMER + BOOKING INFO
    // ----------------------------
    public function customerAndBookingDetails($booking) {
        $this->SetFont('dejavusans', '', 10);

        $startX = $this->GetX();
        $startY = $this->GetY();
        $leftWidth = 115;
        $rightWidth = 70;
        $gap = 2;
        $radius = 3;

        $this->SetFillColor(240, 248, 255);
        $this->SetDrawColor(180, 180, 180);

        $customerText = 
            "Customer Name: " . ucwords(strtolower($booking['customer_name'])) . "\n" .
            "Address: " . ucwords(strtolower($booking['customer_address'])) . "\n\n" .
            "Email: " . $booking['customer_email'] . "\n" .
            "Phone: " . $booking['customer_phone'];

        $bookingText = 
            "Booking No: " . $booking['booking_code'] . "\n" .
            "Booking Date: " . date('d-m-Y', strtotime($booking['booking_date'])) . "\n\n" .
            "Package: " . ucwords(strtolower($booking['package_name'])) . "\n" .
            "Type: " . ucfirst($booking['booking_type']);

        $cellHeight = max(
            $this->getStringHeight($leftWidth - 4, $customerText),
            $this->getStringHeight($rightWidth - 4, $bookingText)
        ) + 8;

        $this->RoundedRect($startX, $startY, $leftWidth, $cellHeight, $radius, '1111', 'DF');
        $this->SetXY($startX + 2, $startY + 2);
        $this->MultiCell($leftWidth - 4, 5, $customerText, 0, 'L', false);

        $this->RoundedRect($startX + $leftWidth + $gap, $startY, $rightWidth, $cellHeight, $radius, '1111', 'DF');
        $this->SetXY($startX + $leftWidth + $gap + 2, $startY + 2);
        $this->MultiCell($rightWidth - 4, 5, $bookingText, 0, 'L', false);

        $this->Ln(8);
    }

    // ----------------------------
    // PACKAGE DETAILS
    // ----------------------------
    public function packageDetails($booking) {
        $this->addSectionTitle('Package & Travel Details');
        $this->SetFont('dejavusans', '', 10);

        $start = date('d-m-Y', strtotime($booking['package_start_date']));
        $end   = date('d-m-Y', strtotime($booking['package_end_date']));
        $interval = (new DateTime($booking['package_start_date']))->diff(new DateTime($booking['package_end_date']))->days;

        $html = '
        <table border="1" cellpadding="6" cellspacing="0" width="100%">
            <tr><td width="40%"><b>Package Name</b></td><td width="60%">' . ucwords($booking['package_name']) . '</td></tr>
            <tr><td><b>Category</b></td><td>' . ucwords($booking['category_name'] ?? '-') . '</td></tr>
            <tr><td><b>Travel Date</b></td><td>' . $start . ' to ' . $end . '</td></tr>
            <tr><td><b>No. of Nights</b></td><td>' . $interval . ' Nights</td></tr>
            <tr><td><b>Guests (Adult/Child/Extra)</b></td><td>' . $booking['package_adults'] . ' Adults / ' . $booking['package_children'] . ' Child / ' . $booking['package_extrabed'] . ' Extra</td></tr>
            <tr><td><b>Room Type</b></td><td>' . ucwords(strtolower($booking['room_type'] ?? '-')) . '</td></tr>
            <tr><td><b>Accommodation</b></td><td>' . ucwords(strtolower($booking['accommodation_type'] ?? '-')) . '</td></tr>
            <tr><td><b>Meal Plan</b></td><td>' . ucwords(strtolower($booking['meal_plan'] ?? '-')) . '</td></tr>
            <tr><td><b>Special Request</b></td><td>' . ucwords(strtolower($booking['special_request'] ?? '-')) . '</td></tr>
        </table>';

        $this->writeHTML($html, true, false, false, false, '');
		
    }

    // ----------------------------
    // PRICING DETAILS (✅ Fixed)
    // ----------------------------
    public function pricingDetails($booking) {
        $this->addSectionTitle('Pricing Details');
        $this->SetFont('dejavusans', '', 10);

        $adults = $booking['package_adults'] ?? 0;
        $child  = $booking['package_children'] ?? 0;
        $extra  = $booking['package_extrabed'] ?? 0;

        $adult_total = $booking['adult_price'] * $adults;
        $child_total = $booking['child_price'] * $child;
        $extra_total = $booking['extra_bed_price'] * $extra;
        $subtotal = $adult_total + $child_total + $extra_total;

        $discount_amt = ($booking['discount'] / 100) * $subtotal;
        $after_discount = $subtotal - $discount_amt;
        $gst_amt = ($booking['gst_included'] === 'yes') ? $booking['gst_amount'] : 0;
       $grand_total = round($after_discount + $gst_amt + (float)$booking['transport_charges'], 2);

        $html = '<table border="1" cellpadding="5" cellspacing="0">';
        $html .= '<tr><td><b>Adult (' . $adults . ' × ' . INR . indian_number_format($booking['adult_price']) . ')</b></td><td>' . INR . indian_number_format($adult_total) . '</td></tr>';

        if ($child > 0 && $child_total > 0)
            $html .= '<tr><td><b>Child (' . $child . ' × ' . INR . indian_number_format($booking['child_price']) . ')</b></td><td>' . INR . indian_number_format($child_total) . '</td></tr>';

        if ($extra > 0 && $extra_total > 0)
            $html .= '<tr><td><b>Extra Bed (' . $extra . ' × ' . INR . indian_number_format($booking['extra_bed_price']) . ')</b></td><td>' . INR . indian_number_format($extra_total) . '</td></tr>';

        $html .= '<tr><td><b>Subtotal</b></td><td>' . INR . indian_number_format($subtotal) . '</td></tr>';

        if ($booking['discount'] > 0)
            $html .= '<tr><td><b>Discount (' . $booking['discount'] . '%)</b></td><td>- ' . INR . indian_number_format($discount_amt) . '</td></tr>';

        if ($gst_amt > 0)
            $html .= '<tr><td><b>GST (' . $booking['gst_rate'] . '%)</b></td><td>' . INR . indian_number_format($gst_amt) . '</td></tr>';

        if ($booking['transport_charges'] > 0)
            $html .= '<tr><td><b>Transport Charges (' . ucwords($booking['vehicle_type'] ?? '-') . ')</b></td><td>' . INR . indian_number_format($booking['transport_charges']) . '</td></tr>';

        $html .= '<tr><td><b>Grand Total</b></td><td><b>' . INR . indian_number_format($grand_total) . '</b></td></tr>';
        $html .= '</table>';

        $this->writeHTML($html, true, false, false, false, '');

        $this->Ln(10);
        $this->addSectionTitle('Terms & Conditions');
        $this->SetFont('dejavusans', '', 9);
        $this->MultiCell(0, 5, 
            "1. Booking once confirmed cannot be canceled.\n2. GST as applicable.\n3. Subject to Ahmedabad jurisdiction only.\n4. Thank you for booking with My Treasure Trip.", 
            0, 'L', false);
    }
}

// ----------------------------
// Generate PDF
// ----------------------------
$booking_id = intval($_GET['id'] ?? 0);
if ($booking_id <= 0) die("Invalid Booking ID");

$query = "SELECT b.*, c.name AS customer_name, c.email AS customer_email, c.phone AS customer_phone, 
                 c.address AS customer_address, p.category_name
          FROM bookings b 
          LEFT JOIN customers c ON b.customer_id = c.id 
          LEFT JOIN package_categories p ON b.category_id = p.id
          WHERE b.id = $booking_id";

$result = $conn->query($query);
if (!$result || $result->num_rows == 0) die("Booking not found");

$booking = $result->fetch_assoc();

$pdf = new BookingOrderPDF();
$pdf->SetMargins(10, 50, 10);
$pdf->SetAutoPageBreak(true, 25);
$pdf->AddPage();
$pdf->customerAndBookingDetails($booking);
$pdf->packageDetails($booking);
//$pdf->hotelDetails($booking_id); // enable if you have hotel table
$pdf->pricingDetails($booking);
$pdf->Output('Booking_' . $booking['booking_code'] . '.pdf', 'I');
?>
