<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include DB connection
include __DIR__ . '/../inc/db.php'; // Adjust path if necessary

header('Content-Type: application/json');

$action = $_REQUEST['action'] ?? '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $id      = $_POST['id'] ?? null;
    $name    = trim($_POST['name'] ?? '');
    $email   = trim($_POST['email'] ?? '');
    $phone   = trim($_POST['phone'] ?? '');
    $address = trim($_POST['address'] ?? '');
    $status  = $_POST['status'] ?? 'active';

    // CREATE
    if ($action === 'Create') {
        if (empty($name) || empty($phone) || empty($address)) {
            echo json_encode(["status" => "error", "message" => "All required fields must be filled."]);
            exit;
        }

        // Check if phone already exists
        $stmt = $conn->prepare("SELECT id FROM customers WHERE phone = ?");
        $stmt->bind_param("s", $phone);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows > 0) {
            echo json_encode(["status" => "error", "message" => "Phone number already exists."]);
            exit;
        }

        $stmt = $conn->prepare("INSERT INTO customers (name, email, phone, address, status) VALUES (?, ?, ?, ?, ?)");
        $stmt->bind_param("sssss", $name, $email, $phone, $address, $status);
        
        if ($stmt->execute()) {
            $customerId = $conn->insert_id;
            echo json_encode([
                'status' => 'success',
                'message' => 'Customer added successfully.',
                'customer' => [
                    'id' => $customerId,
                    'name' => $name,
                    'phone' => $phone
                ]
            ]);
           // echo json_encode(["status" => "success", "message" => "Customer created successfully."]);
        } else {
            echo json_encode(["status" => "error", "message" => "Failed to create customer.", "error" => $stmt->error]);
        }
        exit;
    }

    // UPDATE
    if ($action === 'Update' && $id) {
        if (empty($name) || empty($phone) || empty($address)) {
            echo json_encode(["status" => "error", "message" => "All required fields must be filled."]);
            exit;
        }

        // Check if phone already exists for other records (except the current one being updated)
        $stmt = $conn->prepare("SELECT id FROM customers WHERE phone = ? AND id != ?");
        $stmt->bind_param("si", $phone, $id);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows > 0) {
            echo json_encode(["status" => "error", "message" => "Phone number already exists."]);
            exit;
        }

        $stmt = $conn->prepare("UPDATE customers SET name=?, email=?, phone=?, address=?, status=?, updated_at=CURRENT_TIMESTAMP WHERE id=?");
        $stmt->bind_param("sssssi", $name, $email, $phone, $address, $status, $id);

        if ($stmt->execute()) {
            echo json_encode(["status" => "success", "message" => "Customer updated successfully."]);
        } else {
            echo json_encode(["status" => "error", "message" => "Failed to update customer.", "error" => $stmt->error]);
        }
        exit;
    }

    // DELETE
    elseif ($action === 'Delete' && $id) {
    // Check if customer exists
    $stmt = $conn->prepare("SELECT id FROM customers WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 0) {
        echo json_encode(["status" => "error", "message" => "Customer not found."]);
        exit;
    }
    $stmt->close();

    // Begin transaction for safety
    $conn->begin_transaction();

    try {
        // 1️⃣ Delete related inquiry followups
        $conn->query("DELETE f FROM inquiry_followups f 
                      INNER JOIN inquiries i ON f.inquiry_id = i.id 
                      WHERE i.customer_id = $id");

        // 2️⃣ Delete related inquiries
        $conn->query("DELETE FROM inquiries WHERE customer_id = $id");

        // 3️⃣ Delete related quotation hotels (through quotations)
        $conn->query("DELETE h FROM quotation_hotels h 
                      INNER JOIN quotations q ON h.quotation_id = q.id 
                      WHERE q.customer_id = $id");

        // 4️⃣ Delete related quotations
        $conn->query("DELETE FROM quotations WHERE customer_id = $id");

        // 5️⃣ Delete related bookings
        $conn->query("DELETE FROM bookings WHERE customer_id = $id");

        // 6️⃣ Delete related payments
        $conn->query("DELETE FROM payments WHERE customer_id = $id");

        // 7️⃣ Finally delete the customer
        $stmt = $conn->prepare("DELETE FROM customers WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $stmt->close();

        // Commit all
        $conn->commit();

        echo json_encode(["status" => "success", "message" => "Customer and all related records deleted successfully."]);
    } catch (Exception $e) {
        $conn->rollback();
        echo json_encode(["status" => "error", "message" => "Failed to delete customer.", "error" => $e->getMessage()]);
    }

    exit;
}

    // SELECT SINGLE
    if ($action === 'Select' && $id) {
        $stmt = $conn->prepare("SELECT id, name, email, phone, address, status FROM customers WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows === 0) {
            echo json_encode(["status" => "error", "message" => "Customer not found."]);
            exit;
        }

        $customer = $result->fetch_assoc();
        echo json_encode($customer);
        exit;
    }

    // TOGGLE STATUS
    if ($action === 'ToggleStatus' && $id) {
        $stmt = $conn->prepare("SELECT status FROM customers WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();

        if (!$result) {
            echo json_encode(["status" => "error", "message" => "Customer not found."]);
            exit;
        }

        $new_status = ($result['status'] === 'active') ? 'inactive' : 'active';
        $stmt = $conn->prepare("UPDATE customers SET status = ? WHERE id = ?");
        $stmt->bind_param("si", $new_status, $id);

        if ($stmt->execute()) {
            echo json_encode(["status" => "success", "message" => "Customer status updated.", "new_status" => $new_status]);
        } else {
            echo json_encode(["status" => "error", "message" => "Failed to update status.", "error" => $stmt->error]);
        }
        exit;
    }
}
if ($_SERVER['REQUEST_METHOD'] == 'GET') {
    if ($_GET['action'] == 'getCustomers') {
        // Query to get customers from the database
        $query = "SELECT id, name, phone FROM customers WHERE status = 'active' ORDER BY name ASC";
        $result = $conn->query($query);
        
        // Check if any customers are found
        if ($result->num_rows > 0) {
            $customers = [];
            while ($row = $result->fetch_assoc()) {
                $customers[] = [
                    'id' => $row['id'],
                    'name' => $row['name'],
                    'phone' => $row['phone']
                ];
            }
            echo json_encode(['status' => 'success', 'customers' => $customers]);
        } else {
            echo json_encode(['status' => 'success', 'customers' => []]);
        }
    }
}
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'fetch_customers') {
    $draw = intval($_POST['draw'] ?? 0);
    $start = intval($_POST['start'] ?? 0);
    $length = intval($_POST['length'] ?? 25);
    $colIndex = $_POST['order'][0]['column'] ?? 0;
    $colName = $_POST['columns'][$colIndex]['data'] ?? 'id';
    $colOrder = $_POST['order'][0]['dir'] ?? 'desc';
    $searchValue = mysqli_real_escape_string($conn, $_POST['search']['value'] ?? '');

    // --- Search condition ---
    $searchQuery = "";
    if (!empty($searchValue)) {
        $searchQuery = " AND (c.name LIKE '%$searchValue%' OR c.phone LIKE '%$searchValue%' OR c.email LIKE '%$searchValue%')";
    }

    // --- Count total ---
    $totalResult = mysqli_query($conn, "SELECT COUNT(*) AS allcount FROM customers c");
    $totalRecords = mysqli_fetch_assoc($totalResult)['allcount'];

    // --- Count filtered ---
    $filterResult = mysqli_query($conn, "SELECT COUNT(*) AS allcount FROM customers c WHERE 1 $searchQuery");
    $totalFiltered = mysqli_fetch_assoc($filterResult)['allcount'];

    // --- Main Query (with average discount %) ---
    $query = "
        SELECT 
            c.id,
            c.name,
            c.phone,
            c.email,
            CONCAT_WS(', ', c.address, c.city, c.state) AS address,
            c.status,
            COUNT(b.id) AS total_bookings,
            IFNULL(SUM(b.discount_amount), 0) AS total_discount,
            IFNULL(SUM(b.total_amount), 0) AS total_amount,
            CASE 
                WHEN SUM(b.total_amount) > 0 THEN ROUND((SUM(b.discount_amount) / SUM(b.total_amount)) * 100, 2)
                ELSE 0
            END AS avg_discount_percent
        FROM customers c
        LEFT JOIN bookings b ON c.id = b.customer_id
        WHERE 1 $searchQuery
        GROUP BY c.id
        ORDER BY $colName $colOrder
        LIMIT $start, $length
    ";

    $result = mysqli_query($conn, $query);
    $data = [];

    while ($r = mysqli_fetch_assoc($result)) {
        $statusBtn = '
            <label class="switch">
                <input type="checkbox" class="switch-input" value="'.$r['id'].'" '.($r['status'] === 'active' ? 'checked' : '').' onClick="statusChange(this.value);" />
                <span class="switch-toggle-slider">
                    <span class="switch-on"></span>
                    <span class="switch-off"></span>
                </span>
            </label>';

        $actions = '
            <button class="btn btn-sm btn-primary update" id="'.$r['id'].'"><i class="far fa-edit"></i></button>
            <button class="btn btn-sm btn-danger delete" id="'.$r['id'].'"><i class="far fa-trash-alt"></i></button>
        ';

        $data[] = [
            "name" => htmlspecialchars($r['name']),
            "phone" => htmlspecialchars($r['phone']),
            "email" => htmlspecialchars($r['email']),
            "address" => htmlspecialchars($r['address']),
            "total_bookings" => $r['total_bookings'],
            "avg_discount_percent" => number_format($r['avg_discount_percent'], 2) . '%',
            "total_amount" => number_format($r['total_amount'], 2),
            "status" => $statusBtn,
            "action" => $actions
        ];
    }

    echo json_encode([
        "draw" => $draw,
        "iTotalRecords" => $totalRecords,
        "iTotalDisplayRecords" => $totalFiltered,
        "aaData" => $data
    ]);
    exit;
}
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'getCustomerReports') {
    header('Content-Type: text/html');
    $customer_id = intval($_POST['customer_id']);

    $stmt = $conn->prepare("
        SELECT booking_code, booking_date, total_amount, paid_amount, balance_amount
        FROM bookings
        WHERE customer_id = ?
        ORDER BY booking_date DESC
    ");
    $stmt->bind_param("i", $customer_id);
    $stmt->execute();
    $res = $stmt->get_result();

    if ($res->num_rows === 0) {
        echo "<div class='text-center text-muted p-3'>No bookings found for this customer.</div>";
        exit;
    }

    echo "<table class='table table-sm table-bordered align-middle'>
            <thead class='table-light text-center'>
                <tr>
                    <th>#</th>
                    <th>Booking No</th>
                    <th>Date</th>
                    <th>Total (₹)</th>
                    <th>Paid (₹)</th>
                    <th>Pending (₹)</th>
                </tr>
            </thead>
            <tbody>";
    $i = 1;
    while ($r = $res->fetch_assoc()) {
        echo "<tr class='text-center'>
                <td>{$i}</td>
                <td>{$r['booking_code']}</td>
                <td>" . date('d-m-Y', strtotime($r['booking_date'])) . "</td>
                <td>" . number_format($r['total_amount'], 2) . "</td>
                <td>" . number_format($r['paid_amount'], 2) . "</td>
                <td class='text-danger'>" . number_format($r['balance_amount'], 2) . "</td>
              </tr>";
        $i++;
    }
    echo "</tbody></table>";
    exit;
}

?>
