<?php

namespace App\lib\Html;

use Collective\Html\FormBuilder as IlluminateFormBuilder;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Storage;

class FormBuilder extends IlluminateFormBuilder
{

    /**
     * An array containing the currently opened form groups.
     *
     * @var array
     */
    protected $groupStack = [];

    /**
     * Open up a new HTML form.
     *
     * @param  array $options
     * @return string
     */
    public function open( array $options = [] )
    {
        $options[ "class" ] = isset($options[ "class" ]) ? $options[ "class" ] : 'form-horizontal form-validate-jquery';
        return parent::open( $options );
    }

    /**
     * Open up a new HTML form.
     *
     * @param  array $options
     * @return string
     */
    public function model( $data, array $options = [] )
    {
        $options[ "class" ] = isset($options[ "class" ]) ? $options[ "class" ] : 'form-horizontal form-validate-jquery';
        $options[ "method" ] = isset($options[ "method" ]) ? $options[ "method" ] : 'PATCH';
        return parent::model( $data, $options );
    }

    /**
     * Open up a new fieldset.
     *
     * @param  array $options
     * @return string
     */
    public function openFieldset( $label, array $options = [] )
    {
        $output = '<fieldset><legend class="text-semibold">';
        if ( isset($options[ "icon" ]) )
        {
            $output .= '<i class="' . $options[ "icon" ] . ' position-left"></i>';
        }
        $output .= trans( $label ) . '</legend>';
        return $output;
    }

    /**
     * Close out the last opened fieldset group.
     *
     * @return string
     */
    public function closeFieldset()
    {
        return "</fieldset>";
    }

    /**
     * Create a form input field.
     *
     * @param  string $type
     * @param  string $name
     * @param  string $value
     * @param  array $options
     * @return string
     */
    public function input( $type, $name, $value = null, $options = [] )
    {

        $field_type = isset($options[ 'type' ]) ? $options[ 'type' ] : '';

        if ( $type == "hidden" && $field_type != 'select2')
        {
            return parent::input( $type, $name, $value, $options );
        }
        $options = $options + ["div" => "col-lg-12"];

        unset($options[ 'type' ]);

        if ( $type != "checkbox" && $type != "file" && $field_type != 'select2' )
        {
            $options = $this->appendClassToOptions( 'form-control', $options );
        }
        //Wrap the div
        $html = "%s";
        if ( !isset($options[ "divFalse" ]) )
        {
            $wrapDivClass = isset($options[ "div" ]) ? $options[ "div" ] : "col-lg-9";
            unset($options[ "div" ]);
            $html = '<div class="' . $wrapDivClass . '">%s</div>';
            unset($options[ "div" ]);
        }
        $input = parent::input( $type, $name, $value, $options );

        //Get Error for the field
        $errors = $this->getFormattedErrors( $name );
        //$input = '<span class="checked">' . $input . '</span>' . $errors . '';
        $input = $input . $errors . '';

        //Add form description
        if ( isset($options[ "desc" ]) )
        {
            $input .= '<span class="help-block">' . $options[ "desc" ] . '</span>';
        }
        $html = sprintf( $html, $input );

        // Call the parent input method so that Laravel can handle
        // the rest of the input set up.
        return $html;
    }

    /**
     * Create a form date field.
     *
     * @param  string $type
     * @param  string $name
     * @param  string $value
     * @param  array $options
     * @return string
     */
    public function date( $name, $value = null, $options = [] )
    {
    	$options[ "class" ] = "pickadate";
        $options[ "readonly" ] = "readonly";
        return parent::text( $name, $value, $options );
    }

    /**
     * Create a text input field.
     *
     * @param  string $name
     * @param  string $value
     * @param  array $options
     * @return string
     */
    public function text( $name, $value = null, $options = [] )
    {
        $input = parent::text( $name, $value, $options );
        if ( isset($options[ "required" ]) && ($options[ "required" ] == true) )
        {
            $options[ "aria-required" ] = true;
            //$options["aria-invalid"] = true;
        }
        if ( isset($options[ "icon" ]) )
        {
            $input .= '<div class="form-control-feedback" > <i class="' . $options[ "icon" ] . '"></i></div>';
            unset($options[ "icon" ]);
        }
        // Call the parent input method so that Laravel can handle
        // the rest of the input set up.
        return $input;
    }

    /**
     * Create a button element.
     *
     * @param  string $value
     * @param  array $options
     * @return string
     */
    public function button( $value = null, $options = [] )
    {
        //Convert language variable to value
        $value = !is_null( $value ) ? trans( $value ) : $value;
        if ( isset($options[ "icon" ]) )
        {
            $value .= '<i class="' . $options[ "icon" ] . '"></i>';
            unset($options[ "icon" ]);
        }
        $button = parent::button( $value, $options );
        return $button;
    }

    /**
     * Create a submit button element.
     *
     * @param  string $value
     * @param  array $options
     * @return string
     */
    public function submit_button( $value = null, $options = [] )
    {
        $options = $options + ["type"  => "submit",
                               "class" => "btn btn-primary",
                               "icon"  => "icon-arrow-right14 position-right"];
        return self::button( $value, $options );
    }

    /**
     * Create a cacnel link.
     *
     * @param  string $link
     * @param  array $options
     * @return string
     */
    public function cancel( $link, $options = [] )
    {
        $label = isset($options[ "label" ]) ? $options[ "label" ] : trans( 'common.cancel' );
        $title = isset($options[ "title" ]) ? $options[ "title" ] : trans( 'common.list' );
        $html = "<a href='$link' class='btn btn-default'"
            . "  title='$title'>"
            . "$label </a>";
        return $html;
    }

    /**
     * Create a email input field.
     *
     * @param  string $name
     * @param  string $value
     * @param  array $options
     * @return string
     */
    public function email( $name, $value = null, $options = [] )
    {
        // Call the parent input method so that Laravel can handle
        // the rest of the input set up.
        return parent::email( $name, $value, $options );
    }

    /**
     * Create a email input field.
     *
     * @param  string $name
     * @param  string $value
     * @param  array $options
     * @return string
     */
    public function password( $name, $options = [] )
    {
        // Call the parent input method so that Laravel can handle
        // the rest of the input set up.
        $options[ "autocomplete" ] = "off";
        $input = parent::password( $name, $options );

        if ( isset($options[ "icon" ]) )
        {
            $input .= '<div class="form-control-feedback" > <i class="' . $options[ "icon" ] . '"></i></div>';
            unset($options[ "icon" ]);
        }

        return $input;
    }

    /**
     * Create a file input field.
     *
     * @param  string $name
     * @param  array $options
     * @return string
     */
    public function file( $name, $options = [] )
    {
        $options = $this->appendClassToOptions( 'file-input', $options );
        // Call the parent input method so that Laravel can handle
        // the rest of the input set up.
        $input = parent::file( $name, $options );
        if ( isset($options[ "div" ]) )
        {
            $input = '<div class="' . $options[ "div" ] . '">' . $input . '</div>';
            unset($options[ "div" ]);
        }
        return $input;
    }

    /**
     * Create a file js.
     *
     * @param  string $name
     * @param  array $options
     * @return string
     */
    public function filejs( $options = [] )
    {
        $options = $options + self::getFileUploadDefaultOptions();
        $name = $options[ "selector" ];
        $previewHtml = "";
        if(!empty($options["initialPreview"])){
            foreach($options["initialPreview"] as $img){
                $account_id = Config::get('account.id');
                //$previewHtml .= "\"<img src='/uploads/accounts/$account_id/". $img ."' class='file-preview-image' alt=''>\",";
                $previewHtml .= "\"<img src='". Storage::url($img) ."' class='file-preview-image' alt=''>\",";
            }
        }
        $filejs = <<<filejs
<script type='text/javascript'>
    $(function () {
        // Basic example
        $("$name").fileinput({
            browseClass: '$options[browseClass]',
            browseLabel: '',
            showUpload: $options[showUpload],
            browseIcon: '<i class=\"icon-plus22\"></i> ',
            removeClass: 'btn btn-danger btn-icon',
            removeIcon: '<i class=\"icon-cancel-square\"></i> ',
            layoutTemplates: {
                caption: '<div tabindex=\"-1\" class=\"form-control file-caption {class}\">' + '<span class=\"icon-file-plus kv-caption-icon\"></span><div class=\"file-caption-name\"></div>' + '</div>'
            },
            initialPreview: [
                $previewHtml
            ],
            initialCaption: "No file selected",
        });
    });
</script>
filejs;
        return $filejs;
    }

    public static function getFileUploadDefaultOptions()
    {
        return [
            "selector"    => ".file-input",
            "browseClass" => "btn btn-primary btn-icon",
            "showUpload"  => 'false'
        ];
    }

    /**
     * Create a textarea input field.
     *
     * @param  string $name
     * @param  string $value
     * @param  array $options
     * @return string
     */
    public function textarea( $name, $value = null, $options = [] )
    {
        $options = $this->appendClassToOptions( 'form-control', $options );
        $input = parent::textarea( $name, $value, $options );
        //Get Error for the field
        $errors = $this->getFormattedErrors( $name );
        //$input = '<span class="checked">' . $input . '</span>' . $errors . '';
        $input = $input . $errors . '';
        //Wrap the div
        if ( !isset($options[ "divFalse" ]) )
        {
            $wrapDivClass = isset($options[ "div" ]) ? $options[ "div" ] : "col-lg-12";
            $input = '<div class="' . $wrapDivClass . '">' . $input . '</div>';
            unset($options[ "div" ]);
        }

        // Call the parent input method so that Laravel can handle
        // the rest of the input set up.
        return $input;
    }

    /**
     * Create a select box field.
     *
     * @param  string $name
     * @param  array $list
     * @param  string $selected
     * @param  array $options
     * @return string
     */
    public function select( $name, $list = [], $selected = null, $options = [], array $selectAttributes = [],
        array $optionsAttributes = [],
        array $optgroupsAttributes = [] )
    {
        $options += $this->appendClassToOptions( 'form-control', $options );
        $options += ["div" => "col-lg-12"];
        if ( isset($options[ 'defaultTxt' ]) )
        {
            $list = ['' => $options[ 'defaultTxt' ]] + $list;
        }

        //Get Error for the field
        $errors = $this->getFormattedErrors( $name );
        $input = parent::select( $name, $list, $selected, $options);
        $input = $input . $errors;

        //Wrap the div
        if ( $options[ "div" ] !== false )
        {
            $wrapDivClass = isset($options[ "div" ]) ? $options[ "div" ] : "col-lg-9";
            $input = '<div class="' . $wrapDivClass . '">' . $input . '</div>';
            unset($options[ "div" ]);
        }
        if ( isset($options[ "crud" ]) )
        {
            $input .= '<div class="col-lg-2" style="margin-top: 2%;">
                <ul class="icons-list">
                <li class="dropdown">
                    <a href="#" class="dropdown-toggle" data-toggle="dropdown" aria-expanded="false"><i class="icon-cog3"></i> <span class="caret"></span></a>
                    <ul class="dropdown-menu dropdown-menu-right">
                        <li><a href="javascript:void(0);" onClick="new manageDDCRUD(event,\'' . $options[ "crud" ] . '\',\'add\');"><i class="icon-add"></i>Add New</a></li>
                        <li><a href="javascript:void(0);" onClick="new manageDDCRUD(event,\'' . $options[ "crud" ] . '\',\'edit\');"><i class="icon-pencil7"></i>Edit Selected</a></li>
                        <li><a href="javascript:void(0);" onClick="if(confirm(\'Do you want to Delete?\')){ new manageDDCRUD(event,\'' . $options[ "crud" ] . '\',\'delete\'); }"><i class="icon-trash"></i>Delete Selected</a></li>
                    </ul>
                </li>
            </ul></div>';
        }
        // Call the parent select method so that Laravel can handle
        // the rest of the select set up.
        return $input;
    }

    /**
     * Append the given class to the given options array.
     *
     * @param  string $class
     * @param  array $options
     * @return array
     */
    private function appendClassToOptions( $class, array $options = [] )
    {
        // If a 'class' is already specified, append the 'form-control'
        // class to it. Otherwise, set the 'class' to 'form-control'.
        $options[ 'class' ] = isset($options[ 'class' ]) ? $options[ 'class' ] . ' ' : '';
        $options[ 'class' ] .= $class;

        return $options;
    }

    /**
     * Create a plain form input field.
     *
     * @param  string $type
     * @param  string $name
     * @param  string $value
     * @param  array $options
     * @return string
     */
    public function plainInput( $type, $name, $value = null, $options = [] )
    {
        return parent::input( $type, $name, $value, $options );
    }

    /**
     * Create a plain select box field.
     *
     * @param  string $name
     * @param  array $list
     * @param  string $selected
     * @param  array $options
     * @return string
     */
    public function plainSelect( $name, $list = [], $selected = null, $options = [] )
    {
        return parent::select( $name, $list, $selected, $options );
    }

    /**
     * Determine whether the form element with the given name
     * has any validation errors.
     *
     * @param  string $name
     * @return bool
     */
    private function hasErrors( $name )
    {
        if ( !$this->session->has( 'errors' ) )
        {
            // If the session is not set, or the session doesn't contain
            // any errors, the form element does not have any errors
            // applied to it.
            return false;
        }

        // Get the errors from the session.
        $errors = $this->session->get( 'errors' );

        // Check if the errors contain the form element with the given name.
        // This leverages Laravel's transformKey method to handle the
        // formatting of the form element's name.
        return $errors->has( $this->transformKey( $name ) );
    }

    /**
     * Get the formatted errors for the form element with the given name.
     *
     * @param  string $name
     * @return string
     */
    private function getFormattedErrors( $name )
    {
        if ( !$this->hasErrors( $name ) )
        {
            // If the form element does not have any errors, return
            // an emptry string.
            return '';
        }

        // Get the errors from the session.
        $errors = $this->session->get( 'errors' );

        // Return the formatted error message, if the form element has any.
        return $errors->first( $this->transformKey( $name ), '<p class="help-block error">:message</p>' );
    }

    /**
     * Open a new form group.
     *
     * @param  string $name
     * @param  mixed $label
     * @param  array $options
     * @return string
     */
    public function openGroup( $name, $label = null, $options = [] )
    {
        if(!isset($options['is_form_group']))
        {
            $options = $this->appendClassToOptions( 'form-group col-md-3', $options );
        }
        //Convert language variable to value
        $label = !is_null( $label ) ? __( $label ) : $label;

        // Append the name of the group to the groupStack.
        $this->groupStack[] = $name;

        $lbl_options = [];
        if ( !isset($options[ 'is_class' ]) )
        {
            $options[ "lbl_col_class" ] = isset($options[ "lbl_col_class" ]) ? $options[ "lbl_col_class" ] : "col-lg-12 control-label";
            $lbl_options[ "class" ] = isset($options[ "lbl_class" ]) ? "control-label " . $options[ "lbl_class" ] . " " . $options[ "lbl_col_class" ] : $options[ "lbl_col_class" ];

            if ( isset($options[ "lbl_class" ]) && preg_match("~\brequired\b~",$options[ "lbl_class" ]) ){
                $label = $label.'<span class="required">*</span>';
            }

            unset($options[ "lbl_class" ]);
            unset($options[ "lbl_col_class" ]);
        }

        // If a label is given, we set it up here. Otherwise, we will just
        // set it to an empty string.
        $label = $label ? $this->label( $name, trans( $label ), $lbl_options, false) : '';

        return '<div' . $this->html->attributes( $options ) . '>' . $label;
    }

    /**
     * Close out the last opened form group.
     *
     * @return string
     */
    public function closeGroup()
    {
        // Append the errors to the group and close it out.
        return '</div>';
    }

}
