<?php

namespace App\Modules\Student\Http\Controllers;

use App\Facades\General;
use App\Models\Role;
use App\Modules\Student\Model\Section;
use App\Modules\Student\Model\StudentClass;
use App\Modules\Student\Model\Subject;
use App\Modules\Student\Model\Student;
use App\Modules\Student\Requests\StoreSubject;
use Exception;
use Illuminate\Contracts\View\Factory;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Modules\Student\Http\Requests\StoreStudent;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\View\View;
use Throwable;

class SubjectController extends Controller
{
    /**
     * Module variable
     *
     * @var array
     */
    protected $moduleName;

    /**
     * Class Constructor.
     *
     * @return string
     */
    public function __construct()
    {
        $this->moduleName = $this->getModuleName();
    }

    /**
     * Display a listing of the resource.
     *
     * @param Request $request
     * @return JsonResponse
     * @throws Exception
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            return Subject::getDatatableSubjects();
        }

        General::userLog('SL000801', ['action_module' => $this->moduleName]);
        return View('student::subject.index', ['moduleName' => $this->moduleName]);
    }

    /**
     * Crud method of section
     *
     * @param StoreSubject $request
     * @param int $subjectId
     * @return JsonResponse
     */
    public function crud(StoreSubject $request, $subjectId = 0)
    {
        $subject    = (new Subject())->findById($subjectId);
        $insert     = false;
        $activityId = null;
        if ($subject == null) {
            $subject = new Subject();
            $insert  = true;
        }
        $subject->fill($request->all());
        $subject->account_id = Config::get('account.id');
        if ($insert == true) {
            $subject->created_by = Auth::user()->id;
            $activityId          = 'SL000802';
        } else {
            $subject->updated_by = Auth::user()->id;
            $activityId          = 'SL000803';
        }
        $subject->save();

        General::userLog($activityId, [
            'action_module' => $this->moduleName,
            'parent_id'     => $subject->id,
            'event_data'    => ['name' => $subject->title, 'url' => route($this->moduleName . '.edit', $subject->id)]
        ]);

        return response()->json(["status" => 'added'], 200);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param $subjectId
     * @return JsonResponse
     */
    public function edit($subjectId)
    {
        if (request()->ajax()) {
            $data = Subject::findOrFail($subjectId);
            return response()->json(['result' => $data]);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param Student $subject
     * @return RedirectResponse
     * @throws Exception
     */
    public function destroy($subject)
    {
        if (request()->ajax()) {
            $subject = (new Subject())->findById($subject);
            General::userLog('SL000805', [
                'action_module' => $this->moduleName,
                'parent_id'     => $subject->id,
                'event_data'    => ['name' => $subject->title]
            ]);
            $subject->delete();

            return response()->json(['msg' => trans('locale.success_delete_msg')]);
        }
    }

    /**
     * Display a listing of the resource.
     *
     * @param Request $request
     * @return JsonResponse
     * @throws Exception
     */
    /*public function import(Request $request)
    {
        if($request->isMethod("post")){
            $file = $request->file('file');
            // File Details
            $filename = $file->getClientOriginalName();
            $extension = $file->getClientOriginalExtension();
            $tempPath = $file->getRealPath();
            $fileSize = $file->getSize();
            $mimeType = $file->getMimeType();

             // Valid File Extensions
            $valid_extension = array("csv");

             // 2MB in Bytes
            $maxFileSize = 2097152;
            $exists = 0;
            $total_imports = 0;
            if(in_array(strtolower($extension),$valid_extension)){
                 $data = array_map('str_getcsv', file($tempPath));
                 if($fileSize <= $maxFileSize){
                     foreach($data as $key => $row){
                        if($key == 0) continue;
                        $total_imports++;
                        $subjectModel = new Subject();
                        //Check if student exists?
                        $subjects = $subjectModel->getSubjectByTitle(trim($row[0]));
                        if($subjects){
                            $exists++;
                            continue;
                        }
                        else{
                            $subjectModel->title = trim($row[0]);
                            $subjectModel->account_id = \Illuminate\Support\Facades\Config::get('account.id');
                            $subjectModel->save();
                        }
                     }
                     $msg = "File impported successfully. Run imports over $total_imports subjects, out of which $exists where already exists.";
                     return json_encode(["success" => true, "message" => $msg]);
                 }
                 else{
                     return json_encode(["success" => false, "message" => "File too large. File must be less than 2MB."]);
                 }
            }
            else{
                return json_encode(["success" => false, "message" => "Invalid File Extension."]);
            }
        }
        else{
            return View('student::subject.import', ['moduleName' => $this->moduleName]);
        }
    }*/

    /**
     * Display a listing of the resource.
     *
     * @param Request $request
     * @return false|Factory|View|string
     * @throws Throwable
     */
    public function import(Request $request)
    {
        if ($request->isMethod("post")) {

            $fileResponse = General::uploadImportFile($request, 'file', 'subject');
            if ($fileResponse == 'extension_not_match') {
                return json_encode(["success" => false, "message" => "Invalid File Extension."]);
            }

            if ($fileResponse == 'large_file_size') {
                return json_encode(["success" => false,
                                    "message" => "File too large. File must be less than 2MB."]);
            }

            $importDataArr = self::getDataFromImportedFile()['importDataArr'];
            $preview       = view('student::subject.preview', compact('importDataArr'))->render();

            return json_encode(["success" => true, "data" => $preview]);
        } else {
            return View('student::subject.import', ['moduleName' => $this->moduleName]);
        }
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function importSave(Request $request)
    {
        $importData    = self::getDataFromImportedFile();
        $importDataArr = $importData['importDataArr'];
        $totalData     = $importData['totalData'];
        $totalSaved    = 0;
        foreach ($importDataArr as $key => $data) {

            $subjectModel = new Subject();
            //Check if student exists?
            $subjects = $subjectModel->getSubjectByTitle(trim($data['subject']));
            if ($subjects) {
                continue;
            } else {
                $subjectModel->title      = trim($data['subject']);
                $subjectModel->account_id = Config::get('account.id');
                $subjectModel->save();
            }

            $totalSaved++;
        }

        $msg = "File imported successfully. Run imports over $totalData Subjects, out of which $totalSaved are saved.";
        return response()->json(['message' => $msg], 200);
    }

    /**
     * Return data from imported file
     *
     * @return array
     */
    public function getDataFromImportedFile()
    {
        $fileAccess = General::getImportedFileAccess('subject');

        // Import CSV to preview
        $filepath = storage_path($fileAccess['path'] . "/" . $fileAccess['filename']);
        // Reading file
        $file = fopen($filepath, "r");

        $importDataArr = [];
        $i             = 0;
        $totalData     = 0;
        while (($fileData = fgetcsv($file, 1000, ",")) !== false) {
            // Skip first row
            if ($i == 0) {
                $i++;
                continue;
            }
            if (!empty($fileData[0]) && $fileData[0] != null) {
                $importDataArr[$i]['subject'] = $fileData[0];

                $i++;
            }

            $totalData++;
        }
        fclose($file);

        return [
            'importDataArr' => $importDataArr,
            'totalData'     => $totalData
        ];
    }


    /**
     * Get Module name.
     *
     * @return string
     */
    public function getModuleName()
    {
        return "subject";
    }

}
