<?php
class SalesModel {
    private $conn;

    public function __construct($conn){
        $this->conn = $conn;
    }

    /** CREATE */
    public function saveOrder($data){
        $conn = $this->conn;
        $conn->begin_transaction();

        try {
            // 1) Duplicate bill check
            $bill_no = $conn->real_escape_string($data['bill_no']);
            $checkBill = $conn->query("SELECT id FROM sales WHERE bill_no='$bill_no'");
            if($checkBill && $checkBill->num_rows > 0){
                throw new Exception("Bill No '$bill_no' already exists.");
            }

            // 2) Insert sales
            $agency_id          = intval($data['agency_id']);
            $financial_year_id  = intval($data['financial_year_id']);
            $sale_date          = $conn->real_escape_string($data['sale_date']);
            $total_qty          = intval($data['total_qty']);
            $transport_charge   = floatval($data['transport_charge']);
            $discount           = floatval($data['discount']);
            $total_amount       = floatval($data['total_amount']);
            $remarks            = isset($data['remarks']) ? $conn->real_escape_string($data['remarks']) : '';
            $created_by         = intval($data['created_by']);

            $sqlSale = "INSERT INTO sales
                        (bill_no, agency_id, financial_year_id, sale_date, total_qty, transport_charge, discount, total_amount, remarks, created_by)
                        VALUES ('$bill_no', $agency_id, $financial_year_id, '$sale_date', $total_qty, $transport_charge, $discount, $total_amount, '$remarks', $created_by)";
            if (!$conn->query($sqlSale)) throw new Exception("Error inserting sales: ".$conn->error);

            $sale_id = $conn->insert_id;

            // 3) Insert items (with duplicate-in-payload guard)
            if (empty($data['items']) || !is_array($data['items'])) {
                throw new Exception("No items to insert.");
            }

            $seen = [];
            foreach($data['items'] as $item){
                $newspaper_id = intval($item['newspaper_id']);
                $qty          = intval($item['qty']);
                $rate         = floatval($item['price']);
                $item_remarks = isset($item['remarks']) ? $conn->real_escape_string($item['remarks']) : '';

                if($qty <= 0 || $rate <= 0){
                    throw new Exception("Invalid qty/rate for item.");
                }

                if(isset($seen[$newspaper_id])){
                    throw new Exception("Duplicate newspaper in this order.");
                }
                $seen[$newspaper_id] = true;

                $sqlItem = "INSERT INTO sales_items (sale_id, newspaper_id, qty, rate, remarks)
                            VALUES ($sale_id, $newspaper_id, $qty, $rate, '$item_remarks')";
                if (!$conn->query($sqlItem)) throw new Exception("Error inserting item: ".$conn->error);
            }

            // 4) Commit
            $conn->commit();
            return ['status'=>'success','order_id'=>$sale_id];

        } catch(Exception $e){
            $conn->rollback();
            return ['status'=>'error','message'=>$e->getMessage()];
        }
    }

    /** UPDATE (replace items) */
    public function updateOrder($data){
        $conn = $this->conn;
        $conn->begin_transaction();

        try{
            $sale_id            = intval($data['id']);
            if($sale_id <= 0){ throw new Exception("Invalid order id."); }

            // Ensure order exists
            $exists = $conn->query("SELECT id FROM sales WHERE id=$sale_id");
            if(!$exists || $exists->num_rows === 0){ throw new Exception("Order not found."); }

            // Duplicate bill check (exclude self)
            $bill_no = $conn->real_escape_string($data['bill_no']);
            $checkBill = $conn->query("SELECT id FROM sales WHERE bill_no='$bill_no' AND id<>$sale_id");
            if($checkBill && $checkBill->num_rows > 0){
                throw new Exception("Bill No '$bill_no' already exists.");
            }

            // Recalculate fields (controller already did, but keep safe)
            $agency_id          = intval($data['agency_id']);
            $financial_year_id  = intval($data['financial_year_id']);
            $sale_date          = $conn->real_escape_string($data['sale_date']);
            $total_qty          = intval($data['total_qty']);
            $transport_charge   = floatval($data['transport_charge']);
            $discount           = floatval($data['discount']);
            $total_amount       = floatval($data['total_amount']);
            $remarks            = isset($data['remarks']) ? $conn->real_escape_string($data['remarks']) : '';

            $sqlUpd = "UPDATE sales SET
                        bill_no='$bill_no',
                        agency_id=$agency_id,
                        financial_year_id=$financial_year_id,
                        sale_date='$sale_date',
                        total_qty=$total_qty,
                        transport_charge=$transport_charge,
                        discount=$discount,
                        total_amount=$total_amount,
                        remarks='$remarks'
                       WHERE id=$sale_id";
            if(!$conn->query($sqlUpd)) throw new Exception("Error updating order: ".$conn->error);

            // Replace items (simple & robust)
            if(!$conn->query("DELETE FROM sales_items WHERE sale_id=$sale_id")){
                throw new Exception("Failed clearing items: ".$conn->error);
            }

            if (empty($data['items']) || !is_array($data['items'])) {
                throw new Exception("No items provided.");
            }

            $seen = [];
            foreach($data['items'] as $item){
                $newspaper_id = intval($item['newspaper_id']);
                $qty          = intval($item['qty']);
                $rate         = floatval($item['price']);
                $item_remarks = isset($item['remarks']) ? $conn->real_escape_string($item['remarks']) : '';

                if($qty <= 0 || $rate <= 0){
                    throw new Exception("Invalid qty/rate for item.");
                }

                if(isset($seen[$newspaper_id])){
                    throw new Exception("Duplicate newspaper in this order.");
                }
                $seen[$newspaper_id] = true;

                $sqlItem = "INSERT INTO sales_items (sale_id, newspaper_id, qty, rate, remarks)
                            VALUES ($sale_id, $newspaper_id, $qty, $rate, '$item_remarks')";
                if (!$conn->query($sqlItem)) throw new Exception("Error inserting item: ".$conn->error);
            }

            $conn->commit();
            return ['status'=>'success','order_id'=>$sale_id];

        }catch(Exception $e){
            $conn->rollback();
            return ['status'=>'error','message'=>$e->getMessage()];
        }
    }

    /** DELETE */
    public function deleteOrder($sale_id) {
        $conn = $this->conn;
        $sale_id = intval($sale_id);

        $check = $conn->query("SELECT id FROM sales WHERE id=$sale_id");
        if(!$check || $check->num_rows === 0){
            return ['status'=>'error','message'=>'Order not found'];
        }

        $conn->begin_transaction();
        try {
            if(!$conn->query("DELETE FROM sales_items WHERE sale_id=$sale_id")){
                throw new Exception($conn->error);
            }
            if(!$conn->query("DELETE FROM sales WHERE id=$sale_id")){
                throw new Exception($conn->error);
            }
            $conn->commit();
            return ['status'=>'success','message'=>'Order deleted successfully'];
        } catch(Exception $e){
            $conn->rollback();
            return ['status'=>'error','message'=>$e->getMessage()];
        }
    }

    /** LIST (DataTables) */
    public function getOrders($start, $length, $search, $orderColumn, $orderDir, $financialYearId = null){
        $conn = $this->conn;

        // Map DataTables col index -> column
        $columns = [
            0 => 's.sale_date',
            1 => 's.bill_no',
            2 => 'a.agencyname',
            3 => 's.total_qty',
            4 => 's.total_amount'
        ];

        // Base
        $where = " WHERE 1 ";
        if(!empty($search)){
            $s = $conn->real_escape_string($search);
            $where .= " AND (s.bill_no LIKE '%$s%' OR a.agencyname LIKE '%$s%')";
        }
        if(!empty($financialYearId)){
            $where .= " AND s.financial_year_id=".intval($financialYearId);
        }

        // Count total (no filters by FY for recordsTotal? Typically yes include FY filter if you want filtered dataset)
        $sqlTotal = "SELECT COUNT(*) AS cnt FROM sales s
                     JOIN agency a ON a.id=s.agency_id";
        $totalRes = $conn->query($sqlTotal);
        $recordsTotal = $totalRes ? intval($totalRes->fetch_assoc()['cnt']) : 0;

        // Count filtered
        $sqlFiltered = "SELECT COUNT(*) AS cnt FROM sales s
                        JOIN agency a ON a.id=s.agency_id
                        $where";
        $filteredRes = $conn->query($sqlFiltered);
        $recordsFiltered = $filteredRes ? intval($filteredRes->fetch_assoc()['cnt']) : 0;

        // Order & Limit
        $orderCol = $columns[intval($orderColumn)] ?? 's.sale_date';
        $orderDir = strtoupper($orderDir) === 'DESC' ? 'DESC' : 'ASC';
        $start = max(0, intval($start));
        $length = max(1, intval($length));

        // Data query
        $sql = "SELECT s.id, s.sale_date, s.bill_no, a.agencyname AS agency_name,
                       s.total_qty, s.total_amount
                FROM sales s
                JOIN agency a ON s.agency_id=a.id
                $where
                ORDER BY $orderCol $orderDir
                LIMIT $start, $length";
        $result = $conn->query($sql);

        $data = [];
        if($result){
            while($row = $result->fetch_assoc()){
                $row['sale_date'] = date("d-m-Y", strtotime($row['sale_date']));
                $row['actions'] = '
                    <button class="btn btn-xs btn-primary viewBtn" data-id="'.$row['id'].'" title="View">
                        <i class="tf-icons ti ti-eye"></i>
                    </button>
                    <a href="newspepars_order_edit.php?updateid='.$row['id'].'" class="btn btn-warning btn-xs" title="Edit">
                        <i class="tf-icons ti ti-edit"></i>
                    </a>
                    <button class="btn btn-xs btn-danger deleteBtn" data-id="'.$row['id'].'" title="Delete">
                        <i class="tf-icons ti ti-trash"></i>
                    </button>';
                $data[] = $row;
            }
        }

        return [
            'data' => $data,
            'recordsTotal' => $recordsTotal,
            'recordsFiltered' => $recordsFiltered
        ];
    }

    /** SINGLE (for edit) */
    public function getOrderById($id){
        $conn = $this->conn;
        $id = intval($id);

        $sql = "SELECT s.*, a.agencyname
                FROM sales s
                JOIN agency a ON a.id = s.agency_id
                WHERE s.id=$id
                LIMIT 1";
        $headRes = $conn->query($sql);
        if(!$headRes || $headRes->num_rows === 0) return null;

        $order = $headRes->fetch_assoc();

        $items = [];
        $sqlItems = "SELECT si.id, si.newspaper_id, si.qty, si.rate, si.remarks,
                            n.name AS newspaper_name
                     FROM sales_items si
                     JOIN newspapers n ON n.id=si.newspaper_id
                     WHERE si.sale_id=$id
                     ORDER BY si.id ASC";
        $itemRes = $conn->query($sqlItems);
        if($itemRes){
            while($r = $itemRes->fetch_assoc()){
                $items[] = $r;
            }
        }

        return ['order' => $order, 'items' => $items];
    }
}
